#include "QuaternionControls.h"

#ifdef __APPLE__
#include <OpenGL/gl.h>
#else
#include <GL/gl.h>
#endif

#include <stdlib.h>

#include "Matrix.h"
#include "NumberField.h"
#include "Quaternion.h"
#include "Vector.h"
#include "Utilities.h"

enum {
	FIELD_X = 0,
	FIELD_Y,
	FIELD_Z,
	FIELD_W,
	FIELD_AXIS_X,
	FIELD_AXIS_Y,
	FIELD_AXIS_Z,
	FIELD_ANGLE,
	FIELD_MATRIX_0,
	FIELD_MATRIX_1,
	FIELD_MATRIX_2,
	FIELD_MATRIX_3,
	FIELD_MATRIX_4,
	FIELD_MATRIX_5,
	FIELD_MATRIX_6,
	FIELD_MATRIX_7,
	FIELD_MATRIX_8,
	FIELD_MATRIX_9,
	FIELD_MATRIX_10,
	FIELD_MATRIX_11,
	FIELD_MATRIX_12,
	FIELD_MATRIX_13,
	FIELD_MATRIX_14,
	FIELD_MATRIX_15,
	NUMBER_OF_FIELDS
};

static void (* quaternionChangeCallback)(struct Quaternion);
static void (* axisAngleChangeCallback)(struct Vector, float);
static NumberField numberFields[NUMBER_OF_FIELDS];
static int editNumberField = -1;

static void numberFieldChangeCallback(void * context, float newValue) {
	int numberFieldIndex;
	
	for (numberFieldIndex = 0; numberFieldIndex < NUMBER_OF_FIELDS; numberFieldIndex++) {
		if (context == &numberFields[numberFieldIndex]) break;
	}
	if (numberFieldIndex == FIELD_X ||
	    numberFieldIndex == FIELD_Y ||
	    numberFieldIndex == FIELD_Z ||
	    numberFieldIndex == FIELD_W) {
		(*quaternionChangeCallback)(Quaternion_withValues(numberFields[FIELD_X].value, numberFields[FIELD_Y].value, numberFields[FIELD_Z].value, numberFields[FIELD_W].value));
	} else if (numberFieldIndex == FIELD_AXIS_X ||
	           numberFieldIndex == FIELD_AXIS_Y ||
	           numberFieldIndex == FIELD_AXIS_Z ||
	           numberFieldIndex == FIELD_ANGLE) {
		(*axisAngleChangeCallback)(Vector_withValues(numberFields[FIELD_AXIS_X].value, numberFields[FIELD_AXIS_Y].value, numberFields[FIELD_AXIS_Z].value), numberFields[FIELD_ANGLE].value);
	}
}

void QuaternionControls_init(void (* inQuaternionChangeCallback)(struct Quaternion), void (* inAxisAngleChangeCallback)(struct Vector, float)) {
	quaternionChangeCallback = inQuaternionChangeCallback;
	axisAngleChangeCallback = inAxisAngleChangeCallback;
	
	numberFields[FIELD_X] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_X], 0, 5, 59);
	numberFields[FIELD_Y] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_Y], 0, 80, 59);
	numberFields[FIELD_Z] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_Z], 0, 155, 59);
	numberFields[FIELD_W] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_W], 0, 230, 59);
	
	numberFields[FIELD_AXIS_X] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_AXIS_X], 0, 315, 59);
	numberFields[FIELD_AXIS_Y] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_AXIS_Y], 0, 390, 59);
	numberFields[FIELD_AXIS_Z] = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_AXIS_Z], 0, 465, 59);
	numberFields[FIELD_ANGLE]  = NumberField_create(true, 0.0f, 0.005f, numberFieldChangeCallback, &numberFields[FIELD_ANGLE],  0, 540, 59);
	
	numberFields[FIELD_MATRIX_0]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 1, 625, 59);
	numberFields[FIELD_MATRIX_4]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 2, 700, 59);
	numberFields[FIELD_MATRIX_8]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 3, 775, 59);
	numberFields[FIELD_MATRIX_12] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 4, 850, 59);
	numberFields[FIELD_MATRIX_1]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 1, 625, 41);
	numberFields[FIELD_MATRIX_5]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 2, 700, 41);
	numberFields[FIELD_MATRIX_9]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 3, 775, 41);
	numberFields[FIELD_MATRIX_13] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 4, 850, 41);
	numberFields[FIELD_MATRIX_2]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 1, 625, 23);
	numberFields[FIELD_MATRIX_6]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 2, 700, 23);
	numberFields[FIELD_MATRIX_10] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 3, 775, 23);
	numberFields[FIELD_MATRIX_14] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 4, 850, 23);
	numberFields[FIELD_MATRIX_3]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 5, 625, 5);
	numberFields[FIELD_MATRIX_7]  = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 5, 700, 5);
	numberFields[FIELD_MATRIX_11] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 5, 775, 5);
	numberFields[FIELD_MATRIX_15] = NumberField_create(false, 0.0f, 0.0f, NULL, NULL, 6, 850, 5);
}

void QuaternionControls_setQuaternion(struct Quaternion inQuaternion) {
	Vector axis;
	float angle;
	Matrix matrix;
	
	numberFields[FIELD_X].value = inQuaternion.x;
	numberFields[FIELD_Y].value = inQuaternion.y;
	numberFields[FIELD_Z].value = inQuaternion.z;
	numberFields[FIELD_W].value = inQuaternion.w;
	
	Quaternion_toAxisAngle(inQuaternion, &axis, &angle);
	if (Vector_magnitudeSquared(axis) != 0.0f) {
		Vector_normalize(&axis);
	}
	
	numberFields[FIELD_AXIS_X].value = axis.x;
	numberFields[FIELD_AXIS_Y].value = axis.y;
	numberFields[FIELD_AXIS_Z].value = axis.z;
	numberFields[FIELD_ANGLE].value = angle;
	
	matrix = Quaternion_toMatrix(inQuaternion);
	
	numberFields[FIELD_MATRIX_0].value = matrix.m[0];
	numberFields[FIELD_MATRIX_1].value = matrix.m[1];
	numberFields[FIELD_MATRIX_2].value = matrix.m[2];
	numberFields[FIELD_MATRIX_3].value = matrix.m[3];
	numberFields[FIELD_MATRIX_4].value = matrix.m[4];
	numberFields[FIELD_MATRIX_5].value = matrix.m[5];
	numberFields[FIELD_MATRIX_6].value = matrix.m[6];
	numberFields[FIELD_MATRIX_7].value = matrix.m[7];
	numberFields[FIELD_MATRIX_8].value = matrix.m[8];
	numberFields[FIELD_MATRIX_9].value = matrix.m[9];
	numberFields[FIELD_MATRIX_10].value = matrix.m[10];
	numberFields[FIELD_MATRIX_11].value = matrix.m[11];
	numberFields[FIELD_MATRIX_12].value = matrix.m[12];
	numberFields[FIELD_MATRIX_13].value = matrix.m[13];
	numberFields[FIELD_MATRIX_14].value = matrix.m[14];
	numberFields[FIELD_MATRIX_15].value = matrix.m[15];
}

void QuaternionControls_setEditFieldValue(float value) {
	if (editNumberField != -1) {
		NumberField_setValue(&numberFields[editNumberField], value);
	}
}

float QuaternionControls_getEditFieldValue() {
	if (editNumberField != -1) {
		return numberFields[editNumberField].value;
	}
	return 0.0f;
}

bool QuaternionControls_mouseDown(int x, int y) {
	int numberFieldIndex;
	
	for (numberFieldIndex = 0; numberFieldIndex < NUMBER_OF_FIELDS; numberFieldIndex++) {
		if (NumberField_hitTest(&numberFields[numberFieldIndex], x, y)) {
			editNumberField = numberFieldIndex;
			return true;
		}
	}
	return false;
}

void QuaternionControls_mouseUp(int x, int y) {
	editNumberField = -1;
}

void QuaternionControls_mouseDragged(int x, int y) {
	if (editNumberField != -1) {
		NumberField_mouseDragged(&numberFields[editNumberField], x, y);
	}
}

void QuaternionControls_draw() {
	int numberFieldIndex;
	
	for (numberFieldIndex = 0; numberFieldIndex < NUMBER_OF_FIELDS; numberFieldIndex++) {
		NumberField_draw(&numberFields[numberFieldIndex], numberFieldIndex == editNumberField);
	}
	
	drawGlutString(16, 79, "quat X");
	drawGlutString(91, 79, "quat Y");
	drawGlutString(166, 79, "quat Z");
	drawGlutString(241, 79, "quat W");
	
	drawGlutString(326, 79, "axis X");
	drawGlutString(401, 79, "axis Y");
	drawGlutString(476, 79, "axis Z");
	drawGlutString(551, 79, "angle");
	
	drawGlutString(704, 79, "Quaternion matrix");
}
